import sql from 'mssql';
import { config } from './config.js';

async function createOrder(customerId, bookId, quantity, price, orderDate) {
  try {
    // Connect to the database
    const pool = await sql.connect(config);

    // Begin transaction
    const transaction = new sql.Transaction(pool);
    await transaction.begin();

    try {
      // Check inventory
      const inventoryResult = await transaction
        .request()
        .input('BookID', sql.Int, bookId).query(`
          SELECT Qty
          FROM Inventories
          WHERE BookID = @BookID
        `);

      const currentQty = inventoryResult.recordset[0]?.Qty;

      if (!currentQty || currentQty < quantity) {
        throw new Error('Not enough inventory to fulfill the order');
      }

      // Insert into Orders table
      const orderResult = await transaction
        .request()
        .input('CustomerID', sql.Int, customerId)
        .input('OrderDate', sql.DateTime, orderDate)
        .input('TotalAmount', sql.Decimal(10, 2), price * quantity).query(`
          INSERT INTO Orders (CustomerID, OrderDate, TotalAmount)
          OUTPUT INSERTED.OrderID
          VALUES (@CustomerID, @OrderDate, @TotalAmount)
        `);

      const orderId = orderResult.recordset[0].OrderID;

      // Insert into OrderDetails table
      await transaction
        .request()
        .input('OrderID', sql.Int, orderId)
        .input('BookID', sql.Int, bookId)
        .input('Quantity', sql.Int, quantity)
        .input('Price', sql.Decimal(10, 2), price).query(`
          INSERT INTO OrderDetails (OrderID, BookID, Quantity, Price)
          VALUES (@OrderID, @BookID, @Quantity, @Price)
        `);

      // Update Inventories table
      await transaction
        .request()
        .input('BookID', sql.Int, bookId)
        .input('Quantity', sql.Int, quantity).query(`
          UPDATE Inventories
          SET Qty = Qty - @Quantity
          WHERE BookID = @BookID
        `);

      // Commit transaction
      await transaction.commit();
      console.log('Order created successfully');
    } catch (err) {
      // Rollback transaction on error
      await transaction.rollback();
      throw err;
    }
  } catch (err) {
    console.error('Error creating order:', err);
  }
}

export { createOrder };
